<?php
require_once __DIR__ . '/lib.php';
db_init();

$update = json_decode(file_get_contents('php://input'), true);
if (!$update) { echo "ok"; exit; }

try {
    handle_update($update);
} catch (Throwable $e){
    logx("EX: ".$e->getMessage()." @ ".$e->getFile().":".$e->getLine());
}
echo "ok";

function handle_update($u){
    if (isset($u['message'])) return handle_message($u['message']);
    if (isset($u['callback_query'])) return handle_callback($u['callback_query']);
}

function handle_message($m){
    $chat_id = (int)$m['chat']['id'];
    $from = $m['from'] ?? [];
    $tg_id = (int)($from['id'] ?? 0);

    $user = user_upsert([
        'id'=>$tg_id,
        'chat_id'=>$chat_id,
        'username'=>$from['username'] ?? '',
        'first_name'=>$from['first_name'] ?? '',
    ]);

    // hard block
    if ((int)$user['is_blocked'] === 1 && !is_admin($tg_id)){
        tg_api('sendMessage', ['chat_id'=>$chat_id, 'text'=>"⛔️ دسترسی شما توسط مدیریت محدود شده است."]);
        return;
    }

    // join mandatory check (skip for admin)
    if (!is_admin($tg_id)){
        [$ok,$msg] = must_join_ok($chat_id,$tg_id);
        if (!$ok){
            tg_api('sendMessage', [
                'chat_id'=>$chat_id,
                'text'=>$msg,
                'reply_markup'=>kb_inline([
                    [
                        ['text'=>'✅ عضو شدم', 'callback_data'=>'join_check'],
                        ['text'=>'📢 کانال', 'url'=>'https://t.me/'.ltrim(setting_get('join_channel',''),'@')]
                    ]
                ])
            ]);
            return;
        }
    }

    $text = $m['text'] ?? '';
    $contact = $m['contact'] ?? null;
    $document = $m['document'] ?? null;

    // phone capture
    if (!$user['phone']){
        if ($contact && isset($contact['phone_number'])){
            user_set_phone($tg_id, $contact['phone_number']);
            state_clear($tg_id);
            tg_api('sendMessage', [
                'chat_id'=>$chat_id,
                'text'=>"✅ شماره شما ثبت شد.\n\nحالا از منو یکی رو انتخاب کن 👇",
                'reply_markup'=>main_menu_kb($tg_id)
            ]);
            return;
        }
        tg_api('sendMessage', [
            'chat_id'=>$chat_id,
            'text'=>"👋 سلام!\nبرای ثبت‌نام، لطفاً شماره موبایلت رو ارسال کن.\n\n(روی دکمه زیر بزن)",
            'reply_markup'=>kb_contact()
        ]);
        return;
    }

    // state machine
    $st = state_get($tg_id);
    $state = $st['state'];
    $data = $st['data'];

    if ($text === '/start'){
        tg_api('sendMessage', ['chat_id'=>$chat_id,'text'=>"به فروشگاه 🍎 Apple ID خوش آمدی.\n\nاز منو انتخاب کن 👇",'reply_markup'=>main_menu_kb($tg_id)]);
        return;
    }

    if (is_admin($tg_id) && $text === '/admin'){
        tg_api('sendMessage', ['chat_id'=>$chat_id,'text'=>"🛠 پنل مدیریت",'reply_markup'=>admin_menu_kb()]);
        return;
    
    // Admin quick commands: /block TGID , /unblock TGID , /msg TGID متن...
    if (is_admin($tg_id) && $text){
        if (preg_match('~^/block\s+(\d+)~u', $text, $mm)){
            user_set_block((int)$mm[1], 1);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ کاربر بلاک شد: {$mm[1]}",'reply_markup'=>admin_menu_kb()]);
            return;
        }
        if (preg_match('~^/unblock\s+(\d+)~u', $text, $mm)){
            user_set_block((int)$mm[1], 0);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ کاربر آن‌بلاک شد: {$mm[1]}",'reply_markup'=>admin_menu_kb()]);
            return;
        }
        if (preg_match('~^/msg\s+(\d+)\s+([\s\S]+)$~u', $text, $mm)){
            $target = (int)$mm[1];
            $msg = trim($mm[2]);
            $tu = user_by_tg($target);
            if ($tu){
                tg_api('sendMessage',['chat_id'=>(int)$tu['chat_id'], 'text'=>"📩 پیام از مدیریت:\n\n".$msg]);
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ ارسال شد به {$target}",'reply_markup'=>admin_menu_kb()]);
            } else {
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ کاربر یافت نشد: {$target}",'reply_markup'=>admin_menu_kb()]);
            }
            return;
        }
    }
}

    // -------------------------
    // Admin flows
    // -------------------------
    if (is_admin($tg_id)){
        if ($state === 'admin_add_product_title'){
            $data['title'] = trim($text);
            state_set($tg_id,'admin_add_product_type',$data);
            tg_api('sendMessage', [
                'chat_id'=>$chat_id,
                'text'=>"نوع محصول رو انتخاب کن:",
                'reply_markup'=>kb_inline([
                    [['text'=>'📦 آماده (دارای موجودی)', 'callback_data'=>'adm_prod_type:ready'],['text'=>'🧾 اختصاصی (فرم)', 'callback_data'=>'adm_prod_type:personal']],
                    [['text'=>'⬅️ لغو', 'callback_data'=>'adm_cancel']]
                ])
            ]);
            return;
        }

        if ($state === 'admin_add_product_price'){
            $price = (int)preg_replace('/\D+/','',$text);
            if ($price <= 0){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ قیمت نامعتبره. فقط عدد بفرست."]);
                return;
            }
            $data['price'] = $price;
            state_set($tg_id,'admin_add_product_partner_price',$data);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"قیمت همکاری (تومان) رو بفرست. اگر نداری، 0 بفرست تا برابر قیمت عادی باشد:"]);
            return;
        }

        
        if ($state === 'admin_add_product_partner_price'){
            $pp = (int)preg_replace('/\D+/','',$text);
            if ($pp < 0) $pp = 0;
            $data['price_partner'] = $pp;
            state_set($tg_id,'admin_add_product_desc',$data);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"یک توضیح کوتاه برای محصول بنویس:"]);
            return;
        }

if ($state === 'admin_add_product_desc'){
            $data['description'] = trim($text);
            if (($data['type'] ?? '') === 'ready'){
                // create product first, then prompt stock add
                $pid = admin_product_save($data, []);
                state_clear($tg_id);
                tg_api('sendMessage',[
                    'chat_id'=>$chat_id,
                    'text'=>"✅ محصول آماده اضافه شد.\n\nحالا موجودی (تحویل) رو اضافه کن:",
                    'reply_markup'=>kb_inline([
                        [['text'=>'➕ افزودن موجودی متن', 'callback_data'=>"adm_stock_add_text:{$pid}"]],
                        [['text'=>'➕ افزودن موجودی PDF', 'callback_data'=>"adm_stock_add_pdf:{$pid}"]],
                        [['text'=>'⬅️ بازگشت به پنل', 'callback_data'=>'adm_home']]
                    ])
                ]);
            } else {
                state_set($tg_id,'admin_add_product_payload_personal',$data);
                tg_api('sendMessage',[
                    'chat_id'=>$chat_id,
                    'text'=>"🧾 فرم محصول اختصاصی:\nحالا فیلدهای فرم رو به ترتیب بنویس.\nمثال:\nنام و نام خانوادگی\nایمیل مدنظر\nپسورد مدنظر\n\n(هر خط = یک فیلد)"
                ]);
            }
            return;
        }

        if ($state === 'admin_add_product_payload_personal'){
            $lines = array_values(array_filter(array_map('trim', explode("\n",$text))));
            if (count($lines) < 2){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ حداقل ۲ فیلد بنویس (هر خط یک فیلد)."]);
                return;
            }
            $payload = ['form_fields'=>$lines];
            admin_product_save($data, $payload);
            state_clear($tg_id);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ محصول اختصاصی اضافه شد.",'reply_markup'=>admin_menu_kb()]);
            return;
        }

        if ($state === 'admin_set_join_channel'){
            setting_set('join_channel', trim($text));
            state_clear($tg_id);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ کانال جوین اجباری تنظیم شد.",'reply_markup'=>admin_menu_kb()]);
            return;
        }

        if ($state === 'admin_set_card'){
            $parts = array_map('trim', explode('|',$text));
            if (count($parts) < 2){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ فرمت درست نیست.\nمثال:\n6037... | نام صاحب کارت"]);
                return;
            }
            setting_set('card_number', $parts[0]);
            setting_set('card_holder', $parts[1]);
            state_clear($tg_id);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ کارت‌به‌کارت ذخیره شد.",'reply_markup'=>admin_menu_kb()]);
            return;
        }

        if ($state === 'admin_broadcast_wait'){
            $data = [
                'from_chat_id'=>$chat_id,
                'message_id'=>$m['message_id'],
            ];
            state_set($tg_id,'admin_broadcast_confirm',$data);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"ارسال همگانی به همه کاربران؟",
                'reply_markup'=>kb_inline([
                    [['text'=>'📨 ارسال (Copy)', 'callback_data'=>'adm_bc:copy'],['text'=>'🔁 فوروارد', 'callback_data'=>'adm_bc:fwd']],
                    [['text'=>'⬅️ لغو', 'callback_data'=>'adm_cancel']]
                ])
            ]);
            return;
        }

        if ($state === 'admin_user_msg_wait'){
            $target = (int)($data['target_tg_id'] ?? 0);
            if ($target){
                tg_api('sendMessage', ['chat_id'=>$target,'text'=>"📩 پیام از مدیریت:\n\n".$text]);
                state_clear($tg_id);
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ ارسال شد.",'reply_markup'=>admin_menu_kb()]);
            }
            return;
        }


        if ($state === 'admin_set_partner_price'){
            $pid = (int)($data['product_id'] ?? 0);
            $pp = (int)preg_replace('/\D+/','',$text);
            if ($pp < 0) $pp = 0;
            if ($pp === 0){
                $p = product_get($pid);
                $pp = $p ? (int)$p['price'] : 0;
            }
            product_set_partner_price($pid, $pp);
            state_clear($tg_id);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ قیمت همکاری ذخیره شد.",'reply_markup'=>admin_menu_kb()]);
            return;
        }

        if ($state === 'admin_partner_set'){
            $tid = (int)preg_replace('/\D+/','',$text);
            $urow = $tid ? user_by_tg($tid) : null;
            if (!$urow){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ کاربر پیدا نشد. TG ID درست بفرست."]);
                return;
            }
            state_set($tg_id,'admin_partner_set_confirm',['target_tg'=>$tid]);
            $name = ($urow['username'] ?? '') ? '@'.$urow['username'] : ('TG:'.$urow['tg_id']);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"کاربر: {$name}\n\nچه کاری انجام بدم؟",
                'reply_markup'=>kb_inline([
                    [['text'=>'✅ همکار شود', 'callback_data'=>"adm_partner_set_on:{$tid}"]],
                    [['text'=>'❌ حذف همکاری', 'callback_data'=>"adm_partner_set_off:{$tid}"]],
                    [['text'=>'⬅️ لغو', 'callback_data'=>'adm_cancel']]
                ])
            ]);
            return;
        }
        // NEW: add stock text bulk
        if ($state === 'admin_stock_text'){
            $pid = (int)($data['product_id'] ?? 0);
            if (!$pid){
                state_clear($tg_id);
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ خطا.",'reply_markup'=>admin_menu_kb()]);
                return;
            }
            $raw = trim($text);
            // delimiter: \n---\n (recommended) else each line becomes one item
            $chunks = [];
            if (strpos($raw, "\n---\n") !== false){
                $chunks = array_map('trim', explode("\n---\n", $raw));
            } else {
                $chunks = array_values(array_filter(array_map('trim', explode("\n", $raw))));
            }
            product_stock_add_text_bulk($pid, $chunks);
            $count = product_stock_count($pid);
            state_clear($tg_id);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"✅ موجودی متن اضافه شد.\nموجودی قابل فروش الان: {$count}",
                'reply_markup'=>admin_menu_kb()
            ]);
            return;
        }

        // NEW: add stock pdf item (document)
        if ($state === 'admin_stock_pdf'){
            $pid = (int)($data['product_id'] ?? 0);
            if (!$pid){
                state_clear($tg_id);
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ خطا.",'reply_markup'=>admin_menu_kb()]);
                return;
            }
            if (!$document){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ لطفاً فایل PDF رو به صورت Document ارسال کن."]);
                return;
            }
            $mime = $document['mime_type'] ?? '';
            if ($mime && stripos($mime,'pdf') === false){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ فقط فایل PDF قابل قبوله."]);
                return;
            }
            $file_id = $document['file_id'] ?? '';
            $file_name = $document['file_name'] ?? 'file.pdf';
            if (!$file_id){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ فایل نامعتبر."]);
                return;
            }
            product_stock_add_pdf($pid, $file_id, $file_name);
            $count = product_stock_count($pid);
            state_clear($tg_id);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"✅ موجودی PDF اضافه شد.\nموجودی قابل فروش الان: {$count}",
                'reply_markup'=>admin_menu_kb()
            ]);
            return;
        }
    }

        // NEW: deliver personal order by text
        if ($state === 'admin_deliver_personal_text'){
            $oid = (int)($data['order_id'] ?? 0);
            $o = $oid ? order_get($oid) : null;
            if (!$o){
                state_clear($tg_id);
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ سفارش پیدا نشد.",'reply_markup'=>admin_menu_kb()]);
                return;
            }
            $u = user_by_id((int)$o['user_id']);
            $delivery = trim($text);
            order_set_status($oid,'delivered', order_update_meta($oid, ['delivery_kind'=>'text','delivery_text'=>$delivery]));
            state_clear($tg_id);

            if ($u){
                tg_api('sendMessage',[
                    'chat_id'=>(int)$u['chat_id'],
                    'text'=>"✅ سفارش اختصاصی شما تحویل شد.\n\n🎁 اطلاعات تحویل:\n{$delivery}\n\nشماره سفارش: #{$oid}"
                ]);
            }
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ تحویل انجام شد (#{$oid}).",'reply_markup'=>admin_menu_kb()]);
            return;
        }

        // NEW: deliver personal order by PDF (Document)
        if ($state === 'admin_deliver_personal_pdf'){
            $oid = (int)($data['order_id'] ?? 0);
            $o = $oid ? order_get($oid) : null;
            if (!$o){
                state_clear($tg_id);
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ سفارش پیدا نشد.",'reply_markup'=>admin_menu_kb()]);
                return;
            }
            if (!$document){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ لطفاً فایل PDF رو به صورت Document ارسال کن."]);
                return;
            }
            $mime = $document['mime_type'] ?? '';
            if ($mime && stripos($mime,'pdf') === false){
                tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ فقط فایل PDF قابل قبوله."]);
                return;
            }
            $file_id = $document['file_id'] ?? '';
            $file_name = $document['file_name'] ?? 'AppleID.pdf';
            $u = user_by_id((int)$o['user_id']);

            order_set_status($oid,'delivered', order_update_meta($oid, ['delivery_kind'=>'pdf','file_id'=>$file_id,'file_name'=>$file_name]));
            state_clear($tg_id);

            if ($u){
                tg_api('sendDocument',[
                    'chat_id'=>(int)$u['chat_id'],
                    'document'=>$file_id,
                    'caption'=>"✅ سفارش اختصاصی شما تحویل شد.\nشماره سفارش: #{$oid}\n📄 فایل: {$file_name}"
                ]);
            }
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ تحویل PDF انجام شد (#{$oid}).",'reply_markup'=>admin_menu_kb()]);
            return;
        }

    // -------------------------
    // User flows
    // -------------------------
    if ($state === 'deposit_manual_amount'){
        $amount = (int)preg_replace('/\D+/','',$text);
        if ($amount <= 0){
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ مبلغ نامعتبره. فقط عدد بفرست."]);
            return;
        }
        $data['amount'] = $amount;
        state_set($tg_id,'deposit_manual_receipt',$data);
        tg_api('sendMessage',[
            'chat_id'=>$chat_id,
            'text'=>"✅ مبلغ ثبت شد: ".money($amount)."\n\nحالا عکس رسید کارت‌به‌کارت رو ارسال کن (Photo)."
        ]);
        return;
    }

    if ($state === 'deposit_manual_receipt'){
        if (!isset($m['photo'])){
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ لطفاً عکس رسید رو به صورت Photo ارسال کن."]);
            return;
        }
        $photo = end($m['photo']);
        $file_id = $photo['file_id'] ?? '';
        $amount = (int)($data['amount'] ?? 0);

        $tx_id = wallet_tx_create((int)$user['id'], $amount, 'credit', 'manual', '', 'pending', [
            'file_id'=>$file_id,
            'note'=>'manual deposit',
        ]);

        state_clear($tg_id);
        tg_api('sendMessage',[
            'chat_id'=>$chat_id,
            'text'=>"✅ رسید ثبت شد.\nبعد از تایید مدیریت، کیف پول شما شارژ می‌شود.\n\nکد پیگیری: #TX{$tx_id}",
            'reply_markup'=>main_menu_kb($tg_id)
        ]);

        // notify admin
        $card = setting_get('card_number','');
        $holder = setting_get('card_holder','');
        tg_api('sendPhoto',[
            'chat_id'=>ADMIN_ID,
            'photo'=>$file_id,
            'caption'=>"💳 واریز کارت‌به‌کارت جدید\nمبلغ: ".money($amount)."\nکاربر: @{$user['username']} (TG:{$tg_id})\nTX: #{$tx_id}\nکارت: {$card}\nبه نام: {$holder}",
            'reply_markup'=>kb_inline([
                [['text'=>'✅ تایید', 'callback_data'=>"adm_tx_ok:{$tx_id}"],['text'=>'❌ رد', 'callback_data'=>"adm_tx_no:{$tx_id}"]],
            ])
        ]);
        return;
    }

    // Personal form answers (step-by-step)
    if ($state === 'personal_form'){
        $order_id = (int)($data['order_id'] ?? 0);
        $order = $order_id ? order_get($order_id) : null;
        if (!$order){
            state_clear($tg_id);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ سفارش پیدا نشد.",'reply_markup'=>main_menu_kb($tg_id)]);
            return;
        }
        $meta = json_decode($order['meta'] ?? '{}', true);
        $fields = $meta['form_fields'] ?? [];
        $answers = $meta['answers'] ?? [];
        $step = (int)($meta['step'] ?? 0);

        $answers[$step] = trim($text);
        $step++;

        $meta['answers'] = $answers;
        $meta['step'] = $step;
        order_set_status($order_id, $order['status'], $meta);

        if ($step >= count($fields)){
            // finished
            order_set_status($order_id, 'paid', $meta);
            state_clear($tg_id);

            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"✅ فرم تکمیل شد.\nسفارش شما ثبت شد و به زودی تحویل داده می‌شود.\nشماره سفارش: #{$order_id}",
                'reply_markup'=>main_menu_kb($tg_id)
            ]);

            tg_api('sendMessage',[
                'chat_id'=>ADMIN_ID,
                'text'=>"🧾 سفارش اختصاصی جدید\nOrder #{$order_id}\nUser TG: {$tg_id}\n\nاطلاعات فرم:\n".format_form($fields,$answers)."\n\n(تحویل را دستی برای کاربر ارسال کن)"
            ]);
            return;
        }

        // ask next
        ask_personal_next($chat_id,$tg_id,$order_id);
        return;
    }

    // fallback
    tg_api('sendMessage', [
        'chat_id'=>$chat_id,
        'text'=>"از منو انتخاب کن 👇",
        'reply_markup'=>main_menu_kb($tg_id)
    ]);
}

function handle_callback($cq){
    $data = $cq['data'] ?? '';
    $from = $cq['from'] ?? [];
    $tg_id = (int)($from['id'] ?? 0);
    $chat_id = (int)($cq['message']['chat']['id'] ?? 0);

    $user = user_by_tg($tg_id);
    if (!$user){
        tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'خطا']);
        return;
    }

    if ((int)$user['is_blocked']===1 && !is_admin($tg_id)){
        tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'دسترسی محدود شده']);
        return;
    }

    if ($data === 'join_check'){
        [$ok,$msg] = must_join_ok($chat_id,$tg_id);
        if ($ok){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'✅ تایید شد']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ عضویت تایید شد.\nاز منو انتخاب کن 👇",'reply_markup'=>main_menu_kb($tg_id)]);
        } else {
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'هنوز عضو نیستی']);
        }
        return;
    }

    // main menu
    if ($data === 'menu_products'){
        $rows = [];
        foreach (products_active() as $p){
            $stock = ($p['type']==='ready') ? product_stock_count((int)$p['id']) : null;
            $suffix = ($p['type']==='ready') ? " | موجودی: {$stock}" : "";
            $rows[] = [[
                'text' => ($p['type']==='ready'?'📦 ':'🧾 ') . $p['title'].' — '.money(product_price_for_user($p, $user)).$suffix,
                'callback_data' => 'prod:'.$p['id']
            ]];
        }
        $rows[] = [[ 'text'=>'⬅️ بازگشت', 'callback_data'=>'menu_home' ]];

        tg_api('editMessageText', [
            'chat_id'=>$chat_id,
            'message_id'=>$cq['message']['message_id'],
            'text'=>"🛍 لیست محصولات:",
            'reply_markup'=>kb_inline($rows)
        ]);
        return;
    }

    if ($data === 'menu_wallet'){
        $bal = (int)$user['wallet'];
        tg_api('editMessageText',[
            'chat_id'=>$chat_id,
            'message_id'=>$cq['message']['message_id'],
            'text'=>"💰 کیف پول شما\n\nموجودی: ".money($bal)."\n\nروش شارژ را انتخاب کنید:",
            'reply_markup'=>kb_inline([
                [['text'=>'💳 کارت‌به‌کارت (ارسال رسید)', 'callback_data'=>'dep_manual']],
                [['text'=>'🟡 پرداخت زرین‌پال', 'callback_data'=>'dep_zarinpal']],
                [['text'=>'⬅️ بازگشت', 'callback_data'=>'menu_home']]
            ])
        ]);
        return;
    }

    if ($data === 'dep_manual'){
        $card = setting_get('card_number','');
        $holder = setting_get('card_holder','');
        if ($card===''){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'فعلاً کارت تنظیم نشده']);
            return;
        }
        state_set($tg_id,'deposit_manual_amount',[]);
        tg_api('sendMessage',[
            'chat_id'=>$chat_id,
            'text'=>"💳 کارت برای واریز:\n{$card}\nبه نام: {$holder}\n\nمبلغ واریزی رو فقط به عدد بفرست (مثلاً 250000):"
        ]);
        tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'مبلغ را ارسال کنید']);
        return;
    }

    if ($data === 'dep_zarinpal'){
        if (!ZARINPAL_MERCHANT_ID){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'زرین‌پال تنظیم نشده']);
            return;
        }
        state_set($tg_id,'dep_zp_amount',[]);
        tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"🟡 مبلغ شارژ با زرین‌پال را فقط عدد ارسال کن (تومان):"]);
        tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'مبلغ را ارسال کنید']);
        return;
    }

    if ($data === 'menu_support'){
        $sup = trim(setting_get('support_username',''));
        $txt = "🆘 پشتیبانی\n\n";
        $txt .= ($sup ? "ارتباط: @{$sup}\n" : "پشتیبانی هنوز تنظیم نشده.\n");
        $txt .= "\nهمچنین می‌تونی از بخش «کیف پول» وضعیت رسید رو پیگیری کنی.";
        tg_api('editMessageText',[
            'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
            'text'=>$txt,
            'reply_markup'=>kb_inline([
                [['text'=>'⬅️ بازگشت', 'callback_data'=>'menu_home']]
            ])
        ]);
        return;
    }

    if ($data === 'menu_home'){
        tg_api('editMessageText',[
            'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
            'text'=>"🏠 منوی اصلی",
            'reply_markup'=>main_menu_kb($tg_id)
        ]);
        return;
    }

    // product view
    if (strpos($data,'prod:')===0){
        $pid = (int)substr($data,5);
        $p = product_get($pid);
        if (!$p || (int)$p['is_active']!==1){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'محصول پیدا نشد']);
            return;
        }
        $stock_line = '';
        if ($p['type']==='ready'){
            $stock = product_stock_count($pid);
            $stock_line = "\n📦 موجودی: {$stock}";
        }
        $txt = ($p['type']==='ready'?'📦 ':'🧾 ') . $p['title'] . "\n";
        $txt .= "💳 قیمت: ".money(product_price_for_user($p, $user)).$stock_line."\n\n";
        $txt .= "📝 توضیحات:\n".$p['description']."\n\n";
        $txt .= "✅ روش پرداخت: کیف پول یا زرین‌پال";
        tg_api('editMessageText',[
            'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
            'text'=>$txt,
            'reply_markup'=>kb_inline([
                [['text'=>'✅ خرید با کیف پول', 'callback_data'=>"buy_wallet:{$pid}"]],
                [['text'=>'🟡 خرید با زرین‌پال', 'callback_data'=>"buy_zp:{$pid}"]],
                [['text'=>'⬅️ بازگشت', 'callback_data'=>'menu_products']]
            ])
        ]);
        return;
    }

    // buy with wallet
    if (strpos($data,'buy_wallet:')===0){
        $pid = (int)substr($data,11);
        $p = product_get($pid);
        if (!$p) return;

        // stock check for ready
        if ($p['type']==='ready' && product_stock_count($pid) <= 0){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'ناموجود']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"⛔️ این محصول فعلاً ناموجود است.\nبعداً دوباره تلاش کن.",'reply_markup'=>main_menu_kb($tg_id)]);
            return;
        }

        $price = (int)product_price_for_user($p, $user);
        $bal = (int)$user['wallet'];
        if ($bal < $price){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'موجودی کافی نیست']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"❌ موجودی کافی نیست.\nموجودی: ".money($bal)."\nمبلغ محصول: ".money($price)."\n\nکیف پول رو شارژ کن 👇",'reply_markup'=>main_menu_kb($tg_id)]);
            return;
        }

        // create order and debit wallet
        if ($p['type']==='personal'){
            $payload = json_decode($p['payload'] ?? '{}', true);
            $fields = $payload['form_fields'] ?? [];
            $order_id = order_create((int)$user['id'], $pid, 'personal', $price, [
                'form_fields'=>$fields,
                'answers'=>[],
                'step'=>0
            ]);

            user_wallet_add((int)$user['id'], -$price);
            wallet_tx_create((int)$user['id'], $price, 'debit', 'order', 'ORDER#'.$order_id, 'done', ['product'=>$pid]);
            order_set_status($order_id,'paid');

            state_set($tg_id,'personal_form', ['order_id'=>$order_id]);

            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'پرداخت انجام شد']);
            ask_personal_next($chat_id,$tg_id,$order_id);
            return;
        }

        // READY: reserve stock item and deliver
        $order_id = order_create((int)$user['id'], $pid, 'ready', $price, []);
        user_wallet_add((int)$user['id'], -$price);
        wallet_tx_create((int)$user['id'], $price, 'debit', 'order', 'ORDER#'.$order_id, 'done', ['product'=>$pid]);

        // Reserve one stock item
        $item = product_stock_reserve_one($pid, $order_id);
        if (!$item){
            // out of stock race -> refund wallet
            user_wallet_add((int)$user['id'], $price);
            order_set_status($order_id,'out_of_stock', ['note'=>'no_stock_on_reserve']);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'ناموجود شد']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"⛔️ موجودی تمام شد. مبلغ به کیف پول برگشت.",'reply_markup'=>main_menu_kb($tg_id)]);
            return;
        }

        // Deliver
        deliver_stock_item($chat_id,$tg_id,$order_id,$item);

        tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'✅ خرید موفق']);
        return;
    }

    // buy with zarinpal
    if (strpos($data,'buy_zp:')===0){
        if (!ZARINPAL_MERCHANT_ID){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'زرین‌پال تنظیم نشده']);
            return;
        }
        $pid = (int)substr($data,7);
        $p = product_get($pid);
        if (!$p) return;

        if ($p['type']==='ready' && product_stock_count($pid) <= 0){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'ناموجود']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"⛔️ این محصول فعلاً ناموجود است.",'reply_markup'=>main_menu_kb($tg_id)]);
            return;
        }

        $order_id = $price = (int)product_price_for_user($p, $user);
        $order_id = order_create((int)$user['id'], $pid, $p['type'], $price, []);
        $desc = "Order #{$order_id} - ".$p['title'];
        $callback = BASE_URL . "/zarinpal_verify.php?order_id={$order_id}";
        $req = zarinpal_request((int)$price, $desc, $callback);

        if (!($req['ok'] ?? false)){
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'خطا در ساخت لینک پرداخت']);
            return;
        }
        $payUrl = $req['pay_url'];
        $authority = $req['authority'];

        wallet_tx_create((int)$user['id'], (int)$price, 'credit', 'zarinpal', $authority, 'pending', ['order_id'=>$order_id,'purpose'=>'order']);

        tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'لینک پرداخت ارسال شد']);
        tg_api('sendMessage',[
            'chat_id'=>$chat_id,
            'text'=>"🟡 برای پرداخت روی لینک زیر بزن:\n{$payUrl}\n\nبعد از پرداخت، به صورت خودکار تایید می‌شود.\nشماره سفارش: #{$order_id}",
            'reply_markup'=>main_menu_kb($tg_id),
            'disable_web_page_preview'=>true
        ]);
        return;
    }

    // -------------------------
    // Admin callbacks
    // -------------------------
    if (is_admin($tg_id)){
        if ($data === 'adm_products'){
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"📦 مدیریت محصولات",
                'reply_markup'=>kb_inline([
                    [['text'=>'➕ افزودن محصول', 'callback_data'=>'adm_add_prod']],
                    [['text'=>'📋 لیست محصولات', 'callback_data'=>'adm_list_prod']],
                    [['text'=>'⬅️ بازگشت', 'callback_data'=>'adm_home']]
                ])
            ]);
            return;
        }

        if ($data === 'adm_add_prod'){
            state_set($tg_id,'admin_add_product_title',[]);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"➕ عنوان محصول رو بفرست (مثال: Apple ID آماده - تحویل PDF)"]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'عنوان را ارسال کنید']);
            return;
        }

        if ($data === 'adm_list_prod'){
            $rows = [];
            $pdo = db();
            $st = $pdo->query("SELECT * FROM products ORDER BY id DESC LIMIT 80");
            foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $p){
                $stock = ($p['type']==='ready') ? product_stock_count((int)$p['id']) : null;
                $extra = ($p['type']==='ready') ? " | S:{$stock}" : "";
                $rows[] = [[
                    'text'=>"#{$p['id']} ".($p['is_active']?'✅':'⛔️')." ".$p['title'].$extra,
                    'callback_data'=>"adm_prod_view:{$p['id']}"
                ]];
            }
            $rows[] = [[['text'=>'⬅️ بازگشت','callback_data'=>'adm_products']]];
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"📋 محصولات",
                'reply_markup'=>kb_inline($rows)
            ]);
            return;
        }

        if (strpos($data,'adm_prod_view:')===0){
            $pid = (int)substr($data,14);
            $p = product_get($pid);
            if (!$p) return;
            $stock = ($p['type']==='ready') ? product_stock_count($pid) : null;

            $txt = "#{$p['id']} ".($p['is_active']?'✅ فعال':'⛔️ غیرفعال')."\n";
            $txt .= ($p['type']==='ready'?'📦 آماده (موجودی‌دار)':'🧾 اختصاصی (فرم)')."\n";
            $txt .= "عنوان: {$p['title']}\n";
            $txt .= "قیمت: ".money(product_price_for_user($p, $user))."\n";
            if ($stock !== null) $txt .= "موجودی قابل فروش: {$stock}\n";
            $txt .= "توضیح: {$p['description']}\n";

            $rows = [];
            $rows[] = [['text'=>'💼 ویرایش قیمت همکاری', 'callback_data'=>"adm_prod_partner_price:{$pid}"]];
            if ($p['type']==='ready'){
                $rows[] = [['text'=>'➕ موجودی متن', 'callback_data'=>"adm_stock_add_text:{$pid}"], ['text'=>'➕ موجودی PDF', 'callback_data'=>"adm_stock_add_pdf:{$pid}"]];
            }
            $rows[] = [['text'=>($p['is_active']?'⛔️ غیرفعال‌سازی':'✅ فعال‌سازی'), 'callback_data'=>"adm_prod_toggle:{$pid}"]];
            $rows[] = [['text'=>'🗑 حذف', 'callback_data'=>"adm_prod_del:{$pid}"]];
            $rows[] = [['text'=>'⬅️ بازگشت', 'callback_data'=>'adm_list_prod']];

            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>$txt,
                'reply_markup'=>kb_inline($rows)
            ]);
            return;
        }

        if (strpos($data,'adm_stock_add_text:')===0){
            $pid = (int)substr($data,19);
            state_set($tg_id,'admin_stock_text',['product_id'=>$pid]);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"➕ افزودن موجودی متن\n\n✅ بهترین روش: هر اکانت/تحویل را با جداکننده زیر از هم جدا کن:\n\n---\n\nمثال:\nEmail: a@b.com\nPass: 123\n---\nEmail: c@d.com\nPass: 456\n\nحالا متن رو ارسال کن:"
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'متن را ارسال کنید']);
            return;
        }

        if (strpos($data,'adm_stock_add_pdf:')===0){
            $pid = (int)substr($data,18);
            state_set($tg_id,'admin_stock_pdf',['product_id'=>$pid]);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"➕ افزودن موجودی PDF\n\nحالا فایل PDF رو به صورت Document ارسال کن.\n(هر PDF = 1 موجودی)"
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'PDF را ارسال کنید']);
            return;
        }

        if (strpos($data,'adm_prod_toggle:')===0){
            $pid = (int)substr($data,16);
            $pdo = db();
            $p = product_get($pid);
            if (!$p) return;
            $new = ((int)$p['is_active']===1) ? 0 : 1;
            $st = $pdo->prepare("UPDATE products SET is_active=? WHERE id=?");
            $st->execute([$new,$pid]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'انجام شد']);
            $cq['data'] = "adm_prod_view:{$pid}";
            return handle_callback($cq);
        }

        if (strpos($data,'adm_prod_del:')===0){
            $pid = (int)substr($data,13);
            $pdo = db();
            $pdo->prepare("DELETE FROM product_stock_items WHERE product_id=?")->execute([$pid]);
            $pdo->prepare("DELETE FROM products WHERE id=?")->execute([$pid]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'حذف شد']);
            $cq['data'] = 'adm_list_prod';
            return handle_callback($cq);
        }

        if (strpos($data,'adm_prod_type:')===0){
            $type = substr($data,14);
            $st = state_get($tg_id);
            $d = $st['data'];
            $d['type'] = ($type==='ready') ? 'ready' : 'personal';
            state_set($tg_id,'admin_add_product_price',$d);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"قیمت محصول (تومان) رو فقط عدد بفرست:"]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'قیمت را ارسال کنید']);
            return;
        }

        if ($data === 'adm_stats'){
            $s = stats();
            $txt = "📊 آمار ربات\n\n";
            $txt .= "👥 کاربران: {$s['u']}\n";
            $txt .= "⛔️ بلاک: {$s['b']}\n";
            $txt .= "📦 محصولات: {$s['p']}\n";
            $txt .= "📦 موجودی آماده (کل): {$s['stock']}\n";
            $txt .= "🧾 سفارش‌ها: {$s['o']} (پرداخت‌شده: {$s['paid']} | در انتظار: {$s['pend']})\n";
            $txt .= "💳 رسیدهای در انتظار: {$s['txp']}\n";
            $txt .= "\n⚙️ جوین اجباری: ".(setting_get('join_enabled','0')==='1'?'✅ روشن':'❌ خاموش')."\n";
            $txt .= "📢 کانال: ".(setting_get('join_channel','') ?: '—');
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>$txt,
                'reply_markup'=>kb_inline([
                    [['text'=>'⬅️ بازگشت','callback_data'=>'adm_home']]
                ])
            ]);
            return;
        }

        if ($data === 'adm_join'){
            $enabled = (setting_get('join_enabled','0')==='1');
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"📢 جوین اجباری\n\nوضعیت فعلی: ".($enabled?'✅ روشن':'❌ خاموش')."\nکانال: ".(setting_get('join_channel','') ?: '—'),
                'reply_markup'=>kb_inline([
                    [['text'=>($enabled?'❌ خاموش':'✅ روشن'), 'callback_data'=>'adm_join_toggle']],
                    [['text'=>'✏️ تنظیم کانال', 'callback_data'=>'adm_join_setch']],
                    [['text'=>'⬅️ بازگشت', 'callback_data'=>'adm_home']]
                ])
            ]);
            return;
        }

        if ($data === 'adm_join_toggle'){
            $enabled = (setting_get('join_enabled','0')==='1') ? '0' : '1';
            setting_set('join_enabled',$enabled);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'ذخیره شد']);
            $cq['data']='adm_join'; return handle_callback($cq);
        }

        if ($data === 'adm_join_setch'){
            state_set($tg_id,'admin_set_join_channel',[]);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"آیدی کانال را ارسال کن.\nمثال: @mychannel"]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'کانال را ارسال کنید']);
            return;
        }

        if ($data === 'adm_broadcast'){
            state_set($tg_id,'admin_broadcast_wait',[]);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"پیام همگانی را ارسال کن (متن/عکس/ویدیو/فایل)."]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'پیام را ارسال کنید']);
            return;
        }

        if (strpos($data,'adm_bc:')===0){
            $mode = substr($data,7); // copy|fwd
            $st = state_get($tg_id);
            $d = $st['data'];
            state_clear($tg_id);

            $pdo = db();
            $ids = $pdo->query("SELECT chat_id FROM users WHERE is_blocked=0")->fetchAll(PDO::FETCH_COLUMN);
            $sent=0; $fail=0;
            foreach ($ids as $cid){
                $cid = (int)$cid;
                if ($mode==='fwd'){
                    $r = tg_api('forwardMessage', [
                        'chat_id'=>$cid,
                        'from_chat_id'=>(int)$d['from_chat_id'],
                        'message_id'=>(int)$d['message_id']
                    ]);
                } else {
                    $r = tg_api('copyMessage', [
                        'chat_id'=>$cid,
                        'from_chat_id'=>(int)$d['from_chat_id'],
                        'message_id'=>(int)$d['message_id']
                    ]);
                }
                if (($r['ok'] ?? false)) $sent++; else $fail++;
                usleep(35000);
            }
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ ارسال همگانی انجام شد.\nارسال موفق: {$sent}\nناموفق: {$fail}",'reply_markup'=>admin_menu_kb()]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'انجام شد']);
            return;
        }

        if ($data === 'adm_card'){
            state_set($tg_id,'admin_set_card',[]);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"💳 کارت و نام صاحب کارت را بفرست.\nفرمت:\nشماره کارت | نام صاحب کارت\nمثال:\n6037-xxxx-xxxx-xxxx | علی محمدی"]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'اطلاعات را ارسال کنید']);
            return;
        }

        
        if ($data === 'adm_porders'){
            $orders = orders_personal_open(40);
            $rows = [];
            if (!$orders){
                $rows[] = [[ 'text'=>'— موردی نیست —', 'callback_data'=>'noop' ]];
            } else {
                foreach ($orders as $o){
                    $u = $o['u_un'] ? '@'.$o['u_un'] : ('TG:'.$o['u_tg']);
                    $rows[] = [[
                        'text'=>"#{$o['id']} | ".money($o['amount'])." | {$u}",
                        'callback_data'=>"adm_porder_view:{$o['id']}"
                    ]];
                }
            }
            $rows[] = [[['text'=>'⬅️ بازگشت','callback_data'=>'adm_home']]];
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"🧾 سفارش‌های اختصاصی (باز)\n\nبرای تحویل، یک سفارش را انتخاب کن:",
                'reply_markup'=>kb_inline($rows)
            ]);
            return;
        }

        if (strpos($data,'adm_porder_view:')===0){
            $oid = (int)substr($data,16);
            $o = order_get($oid);
            if (!$o){
                tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'سفارش پیدا نشد']);
                return;
            }
            $u = user_by_id((int)$o['user_id']);
            $meta = json_decode($o['meta'] ?? '{}', true);
            $fields = $meta['form_fields'] ?? [];
            $answers = $meta['answers'] ?? [];
            $formTxt = format_form($fields,$answers);
            $userLine = $u ? ((($u['username']??'')!=='') ? '@'.$u['username'] : 'TG:'.$u['tg_id']) : '—';

            $txt  = "🧾 سفارش اختصاصی #{$o['id']}\n";
            $txt .= "👤 کاربر: {$userLine}\n";
            $txt .= "💳 مبلغ: ".money($o['amount'])."\n";
            $txt .= "📌 وضعیت: {$o['status']}\n\n";
            $txt .= "📝 فرم:\n{$formTxt}\n";

            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>$txt,
                'reply_markup'=>kb_inline([
                    [['text'=>'✅ تحویل (متن)', 'callback_data'=>"adm_porder_del_text:{$oid}"], ['text'=>'📄 تحویل (PDF)', 'callback_data'=>"adm_porder_del_pdf:{$oid}"]],
                    [['text'=>'⬅️ بازگشت', 'callback_data'=>'adm_porders']]
                ])
            ]);
            return;
        }

        if (strpos($data,'adm_porder_del_text:')===0){
            $oid = (int)substr($data,19);
            state_set($tg_id,'admin_deliver_personal_text',['order_id'=>$oid]);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"✅ تحویل سفارش #{$oid}\n\nمتن تحویل را ارسال کن (اطلاعات اکانت/راهنما/لینک‌ها):"
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'متن را ارسال کنید']);
            return;
        }

        if (strpos($data,'adm_porder_del_pdf:')===0){
            $oid = (int)substr($data,18);
            state_set($tg_id,'admin_deliver_personal_pdf',['order_id'=>$oid]);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"📄 تحویل سفارش #{$oid}\n\nفایل PDF را به صورت Document ارسال کن."
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'PDF را ارسال کنید']);
            return;
        }

        if (strpos($data,'adm_prod_partner_price:')===0){
            $pid = (int)substr($data,23);
            state_set($tg_id,'admin_set_partner_price',['product_id'=>$pid]);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"💼 ویرایش قیمت همکاری\n\nقیمت همکاری (تومان) را فقط عدد بفرست.\nاگر 0 بفرستی، برابر قیمت عادی می‌شود."
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'قیمت را ارسال کنید']);
            return;
        }

        if ($data === 'adm_partner'){
            $rows = [
                [['text'=>'➕ تعیین/حذف همکار', 'callback_data'=>'adm_partner_set']],
                [['text'=>'📋 لیست همکارها', 'callback_data'=>'adm_partner_list']],
                [['text'=>'⬅️ بازگشت', 'callback_data'=>'adm_home']]
            ];
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"🤝 پنل همکاری\n\n- همکارها قیمت همکاری می‌بینند و خرید می‌کنند.\n- قیمت همکاری برای هر محصول جداگانه تنظیم می‌شود (از مدیریت محصولات).",
                'reply_markup'=>kb_inline($rows)
            ]);
            return;
        }

        if ($data === 'adm_partner_set'){
            state_set($tg_id,'admin_partner_set',[]);
            tg_api('sendMessage',[
                'chat_id'=>$chat_id,
                'text'=>"➕ تعیین همکار\n\nTG ID کاربر را بفرست.\n(مثال: 123456789)\n\nبعدش انتخاب می‌کنی «همکار شود / حذف همکار»."
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'TG ID را ارسال کنید']);
            return;
        }

        if ($data === 'adm_partner_list'){
            $ps = partners_list(80);
            $rows = [];
            if (!$ps){
                $rows[] = [[ 'text'=>'— همکاری ثبت نشده —', 'callback_data'=>'noop' ]];
            } else {
                foreach ($ps as $urow){
                    $name = ($urow['username'] ?? '') ? '@'.$urow['username'] : ('TG:'.$urow['tg_id']);
                    $rows[] = [[ 'text'=>$name, 'callback_data'=>"adm_partner_view:".$urow['tg_id'] ]];
                }
            }
            $rows[] = [[['text'=>'⬅️ بازگشت','callback_data'=>'adm_partner']]];
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"📋 لیست همکارها",
                'reply_markup'=>kb_inline($rows)
            ]);
            return;
        }

        if (strpos($data,'adm_partner_view:')===0){
            $tid = (int)substr($data,17);
            $urow = user_by_tg($tid);
            if (!$urow){
                tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'یافت نشد']);
                return;
            }
            $name = ($urow['username'] ?? '') ? '@'.$urow['username'] : ('TG:'.$urow['tg_id']);
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"👤 همکار: {$name}\n📲 شماره: ".($urow['phone'] ?: '—')."\n\nمی‌خوای این کاربر از همکاری حذف بشه؟",
                'reply_markup'=>kb_inline([
                    [['text'=>'❌ حذف همکار', 'callback_data'=>"adm_partner_toggle:{$tid}"]],
                    [['text'=>'⬅️ بازگشت', 'callback_data'=>'adm_partner_list']]
                ])
            ]);
            return;
        }

        if (strpos($data,'adm_partner_set_on:')===0){
            $tid = (int)substr($data,19);
            user_set_partner($tid, 1);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'همکار شد']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>'✅ این کاربر همکار شد.', 'reply_markup'=>admin_menu_kb()]);
            state_clear($tg_id);
            return;
        }

        if (strpos($data,'adm_partner_set_off:')===0){
            $tid = (int)substr($data,20);
            user_set_partner($tid, 0);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'حذف شد']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>'✅ همکاری حذف شد.', 'reply_markup'=>admin_menu_kb()]);
            state_clear($tg_id);
            return;
        }

        if (strpos($data,'adm_partner_toggle:')===0){
            $tid = (int)substr($data,19);
            $urow = user_by_tg($tid);
            if (!$urow){ tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'یافت نشد']); return; }
            user_set_partner($tid, 0);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'حذف شد']);
            $cq['data']='adm_partner_list';
            return handle_callback($cq);
        }
if ($data === 'adm_users'){
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"👥 مدیریت کاربران\n\n(نسخه پایه)\nبرای پیام اختصاصی/بلاک/آن‌بلاک فعلاً از این دستورات استفاده کن:\n\n/block 123456789\n/unblock 123456789\n/msg 123456789 سلام\n\n(عدد = TG ID)",
                'reply_markup'=>kb_inline([
                    [['text'=>'⬅️ بازگشت','callback_data'=>'adm_home']]
                ])
            ]);
            return;
        }

        if (strpos($data,'adm_tx_ok:')===0){
            $tx_id = (int)substr($data,10);
            $tx = wallet_tx_get($tx_id);
            if (!$tx || $tx['status']!=='pending') return;
            wallet_tx_set_status($tx_id,'done','manual_ok');
            user_wallet_add((int)$tx['user_id'], (int)$tx['amount']);
            $u = user_by_id((int)$tx['user_id']);
            tg_api('sendMessage',[
                'chat_id'=>(int)$u['chat_id'],
                'text'=>"✅ واریز شما تایید شد.\nمبلغ: ".money($tx['amount'])."\nموجودی جدید: ".money(user_wallet_get((int)$u['id']))
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'تایید شد']);
            return;
        }

        if (strpos($data,'adm_tx_no:')===0){
            $tx_id = (int)substr($data,10);
            $tx = wallet_tx_get($tx_id);
            if (!$tx || $tx['status']!=='pending') return;
            wallet_tx_set_status($tx_id,'rejected','manual_reject');
            $u = user_by_id((int)$tx['user_id']);
            tg_api('sendMessage',[
                'chat_id'=>(int)$u['chat_id'],
                'text'=>"❌ واریز شما رد شد.\nدر صورت نیاز با پشتیبانی تماس بگیرید.\nکد: #TX{$tx_id}"
            ]);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'رد شد']);
            return;
        }

        if ($data === 'adm_home'){
            tg_api('editMessageText',[
                'chat_id'=>$chat_id,'message_id'=>$cq['message']['message_id'],
                'text'=>"🛠 پنل مدیریت",
                'reply_markup'=>admin_menu_kb()
            ]);
            return;
        }

        if ($data === 'adm_cancel'){
            state_clear($tg_id);
            tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'لغو شد']);
            tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"لغو شد.",'reply_markup'=>admin_menu_kb()]);
            return;
        }
    }

    tg_api('answerCallbackQuery',['callback_query_id'=>$cq['id'],'text'=>'OK']);
}

function main_menu_kb($tg_id){
    $rows = [
        [['text'=>'🛍 محصولات Apple ID', 'callback_data'=>'menu_products']],
        [['text'=>'💰 کیف پول', 'callback_data'=>'menu_wallet'], ['text'=>'🆘 پشتیبانی', 'callback_data'=>'menu_support']],
    ];
    if (is_admin($tg_id)){
        $rows[] = [['text'=>'🛠 پنل مدیریت', 'callback_data'=>'adm_home']];
    }
    return kb_inline($rows);
}



function admin_product_save($data, $payload){
    $pdo = db();
    // ensure partner price
    $pp = (int)($data['price_partner'] ?? 0);
    if ($pp <= 0) $pp = (int)($data['price'] ?? 0);

    // price_partner column exists via migration in db_init
    $st = $pdo->prepare("INSERT INTO products(title,type,price,price_partner,description,payload,is_active,created_at) VALUES(?,?,?,?,?,?,?,?)");
    $st->execute([
        (string)($data['title'] ?? ''),
        (string)($data['type'] ?? 'ready'),
        (int)($data['price'] ?? 0),
        (int)$pp,
        (string)($data['description'] ?? ''),
        json_encode($payload, JSON_UNESCAPED_UNICODE),
        1,
        date('c')
    ]);
    return (int)$pdo->lastInsertId();
}

function ask_personal_next($chat_id,$tg_id,$order_id){
    $order = order_get($order_id);
    if (!$order) return;
    $meta = json_decode($order['meta'] ?? '{}', true);
    $fields = $meta['form_fields'] ?? [];
    $answers = $meta['answers'] ?? [];
    $step = (int)($meta['step'] ?? 0);

    if ($step >= count($fields)){
        // finished
        order_set_status($order_id,'paid',$meta);
        tg_api('sendMessage',['chat_id'=>$chat_id,'text'=>"✅ فرم تکمیل شد.\nسفارش شما ثبت شد و به زودی تحویل داده می‌شود.\nشماره سفارش: #{$order_id}",'reply_markup'=>main_menu_kb($tg_id)]);
        tg_api('sendMessage',[
            'chat_id'=>ADMIN_ID,
            'text'=>"🧾 سفارش اختصاصی جدید\nOrder #{$order_id}\nUser TG: {$tg_id}\n\nاطلاعات فرم:\n".format_form($fields,$answers)
        ]);
        state_clear($tg_id);
        return;
    }

    $q = $fields[$step];
    tg_api('sendMessage',[
        'chat_id'=>$chat_id,
        'text'=>"🧾 فرم Apple ID اختصاصی\n\n".($step+1)."/".count($fields)."\n❓ {$q}\n\n(پاسخ رو ارسال کن)"
    ]);
}

function format_form($fields,$answers){
    $out=[];
    foreach ($fields as $i=>$f){
        $a = $answers[$i] ?? '—';
        $out[] = "• {$f}: {$a}";
    }
    return implode("\n",$out);
}

function deliver_stock_item($chat_id, $tg_id, $order_id, $item){
    // item: kind(text|pdf), content, meta
    if (($item['kind'] ?? '') === 'pdf'){
        $file_id = $item['content'] ?? '';
        $file_name = $item['meta']['file_name'] ?? 'AppleID.pdf';
        tg_api('sendDocument',[
            'chat_id'=>$chat_id,
            'document'=>$file_id,
            'caption'=>"✅ خرید شما انجام شد.\nشماره سفارش: #{$order_id}\n\n📄 فایل تحویل: {$file_name}"
        ]);
        order_set_status($order_id,'delivered', ['delivery_kind'=>'pdf','file_name'=>$file_name]);
    } else {
        $text = $item['content'] ?? '✅ خرید شما ثبت شد.';
        tg_api('sendMessage',[
            'chat_id'=>$chat_id,
            'text'=>"✅ خرید انجام شد.\n\n🎁 اطلاعات تحویل:\n{$text}\n\nشماره سفارش: #{$order_id}"
        ]);
        order_set_status($order_id,'delivered', ['delivery_kind'=>'text']);
    }
}

