<?php
require_once __DIR__ . '/config.php';

function logx($msg){
    $line = '['.date('Y-m-d H:i:s').'] '.$msg."\n";
    @file_put_contents(LOG_FILE, $line, FILE_APPEND);
}

function tg_api($method, $params = []){
    $url = "https://api.telegram.org/bot".BOT_TOKEN."/".$method;
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($params, JSON_UNESCAPED_UNICODE),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT => 30,
    ]);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($err) logx("curl_err: $err");
    if ($code >= 400) logx("http_$code: $res");

    $json = json_decode($res, true);
    return $json ?: ['ok'=>false,'error'=>'invalid_json','raw'=>$res];
}

function db(){
    static $pdo = null;
    if ($pdo) return $pdo;
    $pdo = new PDO('sqlite:' . DB_PATH);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    return $pdo;
}

function db_init(){
    $pdo = db();
    $pdo->exec("
    PRAGMA journal_mode=WAL;

    CREATE TABLE IF NOT EXISTS users(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        tg_id INTEGER UNIQUE,
        chat_id INTEGER,
        username TEXT,
        first_name TEXT,
        phone TEXT,
        is_blocked INTEGER DEFAULT 0,
        wallet INTEGER DEFAULT 0,
        created_at TEXT
    );

    CREATE TABLE IF NOT EXISTS settings(
        k TEXT PRIMARY KEY,
        v TEXT
    );

    CREATE TABLE IF NOT EXISTS products(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        title TEXT,
        type TEXT, -- ready|personal
        price INTEGER,
        description TEXT,
        payload TEXT, -- JSON: personal => form_schema
        is_active INTEGER DEFAULT 1,
        created_at TEXT
    );

    -- NEW: stock items for READY products
    CREATE TABLE IF NOT EXISTS product_stock_items(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        product_id INTEGER,
        kind TEXT, -- text|pdf
        content TEXT, -- for text: delivery text ; for pdf: telegram file_id
        meta TEXT, -- JSON: filename, etc
        is_sold INTEGER DEFAULT 0,
        sold_order_id INTEGER DEFAULT NULL,
        created_at TEXT
    );

    CREATE INDEX IF NOT EXISTS idx_stock_product_sold ON product_stock_items(product_id, is_sold);

    CREATE TABLE IF NOT EXISTS orders(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        product_id INTEGER,
        type TEXT, -- ready|personal
        amount INTEGER,
        status TEXT, -- pending|paid|delivered|rejected|out_of_stock
        meta TEXT, -- JSON
        created_at TEXT
    );

    CREATE TABLE IF NOT EXISTS wallet_tx(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        amount INTEGER,
        type TEXT, -- credit|debit
        source TEXT, -- manual|zarinpal|order
        ref TEXT,
        status TEXT, -- pending|done|rejected
        meta TEXT,
        created_at TEXT
    );

    CREATE TABLE IF NOT EXISTS states(
        tg_id INTEGER PRIMARY KEY,
        state TEXT,
        data TEXT,
        updated_at TEXT
    );
    ");

    
    // --- Safe schema migrations (ignore if already exists)
    try { $pdo->exec("ALTER TABLE users ADD COLUMN is_partner INTEGER DEFAULT 0"); } catch (Throwable $e) {}
    try { $pdo->exec("ALTER TABLE products ADD COLUMN price_partner INTEGER DEFAULT 0"); } catch (Throwable $e) {}

    // Backfill partner price if empty/0
    try { $pdo->exec("UPDATE products SET price_partner = price WHERE (price_partner IS NULL OR price_partner=0)"); } catch (Throwable $e) {}
// defaults
    setting_set_default('join_enabled', '0');
    setting_set_default('join_channel', '');
    setting_set_default('card_number', '');
    setting_set_default('card_holder', '');
    setting_set_default('support_username', '');
}

function setting_get($k, $default = ''){
    $pdo = db();
    $st = $pdo->prepare("SELECT v FROM settings WHERE k=? LIMIT 1");
    $st->execute([$k]);
    $v = $st->fetchColumn();
    return ($v === false) ? $default : $v;
}
function setting_set($k, $v){
    $pdo = db();
    $st = $pdo->prepare("INSERT INTO settings(k,v) VALUES(?,?) ON CONFLICT(k) DO UPDATE SET v=excluded.v");
    $st->execute([$k, (string)$v]);
}
function setting_set_default($k,$v){
    $pdo = db();
    $st = $pdo->prepare("INSERT OR IGNORE INTO settings(k,v) VALUES(?,?)");
    $st->execute([$k,(string)$v]);
}

function user_upsert($tg){
    $pdo = db();
    $tg_id = (int)$tg['id'];
    $chat_id = (int)$tg['chat_id'];
    $username = $tg['username'] ?? '';
    $first = $tg['first_name'] ?? '';
    $now = date('c');

    $st = $pdo->prepare("INSERT INTO users(tg_id,chat_id,username,first_name,created_at) VALUES(?,?,?,?,?)
        ON CONFLICT(tg_id) DO UPDATE SET chat_id=excluded.chat_id, username=excluded.username, first_name=excluded.first_name");
    $st->execute([$tg_id,$chat_id,$username,$first,$now]);

    $st2 = $pdo->prepare("SELECT * FROM users WHERE tg_id=? LIMIT 1");
    $st2->execute([$tg_id]);
    return $st2->fetch(PDO::FETCH_ASSOC);
}

function user_by_tg($tg_id){
    $pdo = db();
    $st = $pdo->prepare("SELECT * FROM users WHERE tg_id=? LIMIT 1");
    $st->execute([(int)$tg_id]);
    return $st->fetch(PDO::FETCH_ASSOC);
}

function user_by_id($id){
    $pdo = db();
    $st = $pdo->prepare("SELECT * FROM users WHERE id=? LIMIT 1");
    $st->execute([(int)$id]);
    return $st->fetch(PDO::FETCH_ASSOC);
}

function user_set_phone($tg_id, $phone){
    $pdo = db();
    $st = $pdo->prepare("UPDATE users SET phone=? WHERE tg_id=?");
    $st->execute([(string)$phone, (int)$tg_id]);
}

function user_set_block($tg_id, $blocked){
    $pdo = db();
    $st = $pdo->prepare("UPDATE users SET is_blocked=? WHERE tg_id=?");
    $st->execute([(int)$blocked, (int)$tg_id]);
}

function user_wallet_add($user_id, $amount){
    $pdo = db();
    $st = $pdo->prepare("UPDATE users SET wallet = wallet + ? WHERE id=?");
    $st->execute([(int)$amount, (int)$user_id]);
}

function user_wallet_get($user_id){
    $pdo = db();
    $st = $pdo->prepare("SELECT wallet FROM users WHERE id=?");
    $st->execute([(int)$user_id]);
    return (int)$st->fetchColumn();
}

function state_get($tg_id){
    $pdo = db();
    $st = $pdo->prepare("SELECT state,data FROM states WHERE tg_id=?");
    $st->execute([(int)$tg_id]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    if (!$row) return ['state'=>'','data'=>[]];
    $data = json_decode($row['data'] ?? '[]', true);
    return ['state'=>$row['state'] ?? '', 'data'=> is_array($data)?$data:[]];
}
function state_set($tg_id, $state, $data = []){
    $pdo = db();
    $st = $pdo->prepare("INSERT INTO states(tg_id,state,data,updated_at) VALUES(?,?,?,?)
        ON CONFLICT(tg_id) DO UPDATE SET state=excluded.state, data=excluded.data, updated_at=excluded.updated_at");
    $st->execute([(int)$tg_id, (string)$state, json_encode($data, JSON_UNESCAPED_UNICODE), date('c')]);
}
function state_clear($tg_id){
    $pdo = db();
    $st = $pdo->prepare("DELETE FROM states WHERE tg_id=?");
    $st->execute([(int)$tg_id]);
}

function kb_inline($rows){
    return ['inline_keyboard'=>$rows];
}

function kb_contact(){
    return [
        'keyboard' => [
            [[ 'text' => '📲 ارسال شماره موبایل', 'request_contact' => true ]]
        ],
        'resize_keyboard' => true,
        'one_time_keyboard' => true
    ];
}

function is_admin($tg_id){
    return ((int)$tg_id === (int)ADMIN_ID);
}

function must_join_ok($chat_id, $user_tg_id){
    if (setting_get('join_enabled','0') !== '1') return [true,''];
    $channel = trim(setting_get('join_channel',''));
    if ($channel === '') return [true,''];
    $ch = ltrim($channel,'@');

    $res = tg_api('getChatMember', ['chat_id'=>'@'.$ch, 'user_id'=>(int)$user_tg_id]);
    if (!($res['ok'] ?? false)) {
        // fail-open if bot cannot check
        return [true,''];
    }
    $status = $res['result']['status'] ?? '';
    $ok = in_array($status, ['member','administrator','creator'], true);
    if ($ok) return [true,''];
    $msg = "⛔️ برای استفاده از ربات، اول باید عضو کانال شوید.\n\n👉 @{$ch}\n\nبعد از عضویت، روی «✅ عضو شدم» بزنید.";
    return [false, $msg];
}

function money($n){
    return number_format((int)$n) . ' ' . CURRENCY_LABEL;
}

function products_active(){
    $pdo = db();
    $st = $pdo->query("SELECT * FROM products WHERE is_active=1 ORDER BY id DESC");
    return $st->fetchAll(PDO::FETCH_ASSOC);
}

function product_get($id){
    $pdo = db();
    $st = $pdo->prepare("SELECT * FROM products WHERE id=? LIMIT 1");
    $st->execute([(int)$id]);
    return $st->fetch(PDO::FETCH_ASSOC);
}

function product_stock_count($product_id){
    $pdo = db();
    $st = $pdo->prepare("SELECT COUNT(*) FROM product_stock_items WHERE product_id=? AND is_sold=0");
    $st->execute([(int)$product_id]);
    return (int)$st->fetchColumn();
}

function product_stock_add_text_bulk($product_id, $chunks){
    $pdo = db();
    $pdo->beginTransaction();
    $st = $pdo->prepare("INSERT INTO product_stock_items(product_id,kind,content,meta,is_sold,created_at) VALUES(?,?,?,?,0,?)");
    foreach ($chunks as $c){
        $c = trim((string)$c);
        if ($c==='') continue;
        $st->execute([(int)$product_id,'text',$c,json_encode([],JSON_UNESCAPED_UNICODE),date('c')]);
    }
    $pdo->commit();
}

function product_stock_add_pdf($product_id, $file_id, $file_name=''){
    $pdo = db();
    $st = $pdo->prepare("INSERT INTO product_stock_items(product_id,kind,content,meta,is_sold,created_at) VALUES(?,?,?,?,0,?)");
    $meta = ['file_name'=>$file_name];
    $st->execute([(int)$product_id,'pdf',(string)$file_id,json_encode($meta,JSON_UNESCAPED_UNICODE),date('c')]);
}

function product_stock_reserve_one($product_id, $order_id){
    // reserve first available item atomically-ish in SQLite
    $pdo = db();
    $pdo->beginTransaction();
    $st = $pdo->prepare("SELECT id,kind,content,meta FROM product_stock_items WHERE product_id=? AND is_sold=0 ORDER BY id ASC LIMIT 1");
    $st->execute([(int)$product_id]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    if (!$row){
        $pdo->rollBack();
        return null;
    }
    $upd = $pdo->prepare("UPDATE product_stock_items SET is_sold=1, sold_order_id=? WHERE id=? AND is_sold=0");
    $upd->execute([(int)$order_id,(int)$row['id']]);
    if ($upd->rowCount() !== 1){
        $pdo->rollBack();
        return null;
    }
    $pdo->commit();
    $row['meta'] = json_decode($row['meta'] ?? '{}', true);
    return $row;
}

function order_create($user_id, $product_id, $type, $amount, $meta = []){
    $pdo = db();
    $st = $pdo->prepare("INSERT INTO orders(user_id,product_id,type,amount,status,meta,created_at) VALUES(?,?,?,?,?,?,?)");
    $st->execute([(int)$user_id,(int)$product_id,(string)$type,(int)$amount,'pending',json_encode($meta, JSON_UNESCAPED_UNICODE),date('c')]);
    return (int)$pdo->lastInsertId();
}

function order_get($id){
    $pdo = db();
    $st = $pdo->prepare("SELECT * FROM orders WHERE id=? LIMIT 1");
    $st->execute([(int)$id]);
    return $st->fetch(PDO::FETCH_ASSOC);
}

function order_set_status($id, $status, $meta = null){
    $pdo = db();
    if ($meta === null){
        $st = $pdo->prepare("UPDATE orders SET status=? WHERE id=?");
        $st->execute([(string)$status,(int)$id]);
    } else {
        $st = $pdo->prepare("UPDATE orders SET status=?, meta=? WHERE id=?");
        $st->execute([(string)$status,json_encode($meta, JSON_UNESCAPED_UNICODE),(int)$id]);
    }
}

function wallet_tx_create($user_id, $amount, $type, $source, $ref='', $status='pending', $meta=[]){
    $pdo = db();
    $st = $pdo->prepare("INSERT INTO wallet_tx(user_id,amount,type,source,ref,status,meta,created_at) VALUES(?,?,?,?,?,?,?,?)");
    $st->execute([(int)$user_id,(int)$amount,(string)$type,(string)$source,(string)$ref,(string)$status,json_encode($meta, JSON_UNESCAPED_UNICODE),date('c')]);
    return (int)$pdo->lastInsertId();
}

function wallet_tx_get($id){
    $pdo = db();
    $st = $pdo->prepare("SELECT * FROM wallet_tx WHERE id=? LIMIT 1");
    $st->execute([(int)$id]);
    return $st->fetch(PDO::FETCH_ASSOC);
}

function wallet_tx_set_status($id, $status, $ref=''){
    $pdo = db();
    $st = $pdo->prepare("UPDATE wallet_tx SET status=?, ref=? WHERE id=?");
    $st->execute([(string)$status,(string)$ref,(int)$id]);
}

function stats(){
    $pdo = db();
    $u = (int)$pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
    $b = (int)$pdo->query("SELECT COUNT(*) FROM users WHERE is_blocked=1")->fetchColumn();
    $p = (int)$pdo->query("SELECT COUNT(*) FROM products")->fetchColumn();
    $o = (int)$pdo->query("SELECT COUNT(*) FROM orders")->fetchColumn();
    $paid = (int)$pdo->query("SELECT COUNT(*) FROM orders WHERE status IN ('paid','delivered')")->fetchColumn();
    $pend = (int)$pdo->query("SELECT COUNT(*) FROM orders WHERE status='pending'")->fetchColumn();
    $txp = (int)$pdo->query("SELECT COUNT(*) FROM wallet_tx WHERE status='pending'")->fetchColumn();
    $stock = (int)$pdo->query("SELECT COUNT(*) FROM product_stock_items WHERE is_sold=0")->fetchColumn();
    return compact('u','b','p','o','paid','pend','txp','stock');
}

function zarinpal_request($amount_toman, $description, $callback_url){
    $data = [
        "merchant_id" => ZARINPAL_MERCHANT_ID,
        "amount" => (int)$amount_toman,
        "callback_url" => $callback_url,
        "description" => $description,
        "metadata" => [ "mobile" => "", "email" => "" ]
    ];

    $ch = curl_init(ZARINPAL_REQUEST_URL);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($data, JSON_UNESCAPED_UNICODE),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json','Accept: application/json'],
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT => 30,
    ]);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    if ($err) return ['ok'=>false,'error'=>$err];

    $json = json_decode($res, true);
    if (!is_array($json)) return ['ok'=>false,'error'=>'invalid_json','raw'=>$res];

    $code = $json['data']['code'] ?? null;
    $authority = $json['data']['authority'] ?? null;
    if ((int)$code === 100 && $authority){
        return [
            'ok'=>true,
            'authority'=>$authority,
            'pay_url'=>ZARINPAL_STARTPAY.$authority
        ];
    }
    return ['ok'=>false,'error'=>'zarinpal_request_failed','raw'=>$json];
}




function orders_personal_open($limit=30){
    $pdo = db();
    $st = $pdo->prepare("SELECT o.*, u.tg_id AS u_tg, u.chat_id AS u_chat, u.username AS u_un
                         FROM orders o
                         JOIN users u ON u.id=o.user_id
                         WHERE o.type='personal' AND o.status IN ('paid','pending')
                         ORDER BY o.id DESC
                         LIMIT ?");
    $st->bindValue(1, (int)$limit, PDO::PARAM_INT);
    $st->execute();
    return $st->fetchAll(PDO::FETCH_ASSOC);
}

function order_update_meta($order_id, $patch){
    $order = order_get($order_id);
    if (!$order) return false;
    $meta = json_decode($order['meta'] ?? '{}', true);
    if (!is_array($meta)) $meta = [];
    foreach ((array)$patch as $k=>$v){
        $meta[$k]=$v;
    }
    order_set_status($order_id, $order['status'], $meta);
    return $meta;
}




function user_set_partner($tg_id, $is_partner){
    $pdo = db();
    $st = $pdo->prepare("UPDATE users SET is_partner=? WHERE tg_id=?");
    $st->execute([(int)$is_partner, (int)$tg_id]);
}

function is_partner_user($user_row){
    return (int)($user_row['is_partner'] ?? 0) === 1;
}

function product_price_for_user($product, $user_row){
    $normal = (int)($product['price'] ?? 0);
    $partner = (int)($product['price_partner'] ?? 0);
    if ($partner <= 0) $partner = $normal;
    return is_partner_user($user_row) ? $partner : $normal;
}

function partners_list($limit=80){
    $pdo = db();
    $st = $pdo->prepare("SELECT * FROM users WHERE is_partner=1 ORDER BY id DESC LIMIT ?");
    $st->bindValue(1,(int)$limit, PDO::PARAM_INT);
    $st->execute();
    return $st->fetchAll(PDO::FETCH_ASSOC);
}

function product_set_partner_price($product_id, $price_partner){
    $pdo = db();
    $st = $pdo->prepare("UPDATE products SET price_partner=? WHERE id=?");
    $st->execute([(int)$price_partner,(int)$product_id]);
}

